/*
	$DOCFILE:CHAT.C

	Copyright (C) 1999 by Datalight, Inc.
	All Rights Reserved.
	Datalight Confidential and Proprietary.
		
	A TCP based chat application. A server is started on the default chat port.
	All connections made to this server as well as those made by the local user
	to other servers, are put in a list. Whatever data the local user enters is
	send to all the connections in this list (When he hits Enter). Any data
	received from any of these listed connections is displayed on the screen.

	See the program for documentation on special keys (Alt-H).

	This program and is functions is single-threaded and non-reentrant and should
	be used as such.

	$DOCHISTORY
	06/29/1999 jmb added copyright header
*/

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#include <process.h>
#include "compiler.h"
#include "capi.h"
#include "chat.pt"

#define CHAT_PORT 33
#define MAX_CONS 30
#define BUF_SIZE 200

static int rgiSocks[MAX_CONS];// all the sockets (passive and active)
static char rgszNames[MAX_CONS][80];//Names associated with connections
static NET_ADDR sNetAddr;//for general use

static char rgcKeyBuf[BUF_SIZE];//key input buffer
static int iKeyCount = 0;//number of characters in above buffer


/*
Sits in loop and look for both user input or network input.
*/
void main(void)
{

	int iListenSock;//listening socket, copied to rgiSocks once connection made

	//general variables
	char rgcBuf[BUF_SIZE];
	int iIndex, iLength;
	char cCh;

	printf("Sockets Chat client\n");
	printf("Copyright (C) 1999 Datalight, Inc.\nAll Rights Reserved\n\n");

	for (iIndex = 0; iIndex < MAX_CONS; iIndex++)
		rgiSocks[iIndex] = 0;
	iListenSock = StartListen(CHAT_PORT);

	Aprintf("Press Alt-H for help\n");

	while (1)
	{
		//server part - see if new con was opened
		iLength = ReadSocket(iListenSock, rgcBuf, 0, 0, 0);
		if (iNetErrNo != ERR_NOT_NET_CONN && iNetErrNo != ERR_NOT_ESTAB)
		{
			//connection made and established
			if ((iIndex = GetFirstOpen()) != -1)
			{
				//get unused entry
				rgiSocks[iIndex] = iListenSock;
				GetPeerAddress(iListenSock, &sNetAddr);
				sprintf(rgszNames[iIndex],"%s",WriteName(&sNetAddr));
				Aprintf("Connection made by %s\n", rgszNames[iIndex]);
				iLength = WriteSocket(rgiSocks[iIndex], "(server)Hello!", 14, 0);
				if (iLength < 0)
					Aprintf("Error on NetWrite hello: %s\n",Err(iNetErrNo));
			}
			else
				ReleaseSocket(iListenSock);
			iListenSock = StartListen(CHAT_PORT);
			if (iListenSock == 0)
				goto exit;
		}
		else
		{
			static int LastErr = 1000;
			if (iNetErrNo != LastErr)
			{
				LastErr = iNetErrNo;
				printf("Error = %d %s\n",iNetErrNo,Err(iNetErrNo));
			}
		}

		//all sockets - see if data is available
		ResetEnum();
		while ((iIndex = GetNextIndex()) != -1)
		{
			iLength = ReadSocket(rgiSocks[iIndex], rgcBuf, BUF_SIZE, 0, 0);
			if (iLength <= 0)
			{
				if (iNetErrNo == ERR_WOULD_BLOCK || iNetErrNo == ERR_NOT_ESTAB)
					continue;	/* nothing to read or not established yet*/
				if (iLength == 0 && iNetErrNo == 0)
					Aprintf("Peer (%s) has closed connection\n", rgszNames[iIndex]);
				else
					Aprintf("Error on ReadSocket from %s: %s - closing\n",rgszNames[iIndex], Err(iNetErrNo));
				ReleaseSocket(rgiSocks[iIndex]);
				rgiSocks[iIndex] = 0;
			}
			else//display the data (without color for now)
			{
				rgcBuf[iLength] = 0;
				//textattr(0x3d);//rgiSocks[iIndex] | 8);//make it a bright color corresponding to this socket
				//cprintf("xx");
				Aprintf("%s: %s\n", rgszNames[iIndex], rgcBuf);
				//textattr(7);
			}
		}
		//keyboard: que data and send on all connections
		if (kbhit())
		{
			switch(cCh = getche())
			{
			case 0:				/* function key */
				printf("\n");
				switch (cCh = getch())
				{
				default:
					printf("Undefined function key: %d\n",cCh);
					//fall thru
				case 35:		/* alt H */
					printf("Alt-C	Close connection\n"
						"Alt-N	New connection\n"
						"Alt-H	Help\n"
						"Alt-L	List connections\n"
						"Alt-X	eXit\n");
					break;

				case 45:		/* alt X */
					goto exit;
				case 49:        /* alt N */
					//make a new connection
					if ((iIndex = GetFirstOpen()) != -1)
					{
						printf("Enter host to connect to:");
						if ((rgiSocks[iIndex] = ConnectTo(gets(rgcBuf))) != 0)
							sprintf(rgszNames[iIndex],"%s", rgcBuf);
					}
					else
						printf("Max number of connections in use\n");
					break;
				case 38:		/* alt L */
					printf("List of all connections\n");
					ResetEnum();
					while ((iIndex = GetNextIndex()) != -1)
						printf("Connection no %d descriptor:%u name %s \n", iIndex, rgiSocks[iIndex], rgszNames[iIndex]);
					printf("List end\n");
					break;
				case 46:		/* alt C */ //close a connection
					printf("Enter connection no to close:");
					iIndex = atoi(gets(rgcBuf));
					if (iIndex < 0 || iIndex > MAX_CONS)
						printf("OUT of range:%d\n", iIndex);
					else if (rgiSocks[iIndex])
					{
						ReleaseSocket(rgiSocks[iIndex]);
						rgiSocks[iIndex] = 0;
						printf("Closed %s\n", rgszNames[iIndex]);
					}
					else
						printf("Connection %d not open\n", iIndex);
					break;

				}
				Aprintf("");//just to reprint any edited stuff
				break;
			default:
				if (iKeyCount < BUF_SIZE)
				{
					rgcKeyBuf[iKeyCount++] = cCh;
					break;
				}
				//fall thru
			case '\r':
				printf ("\n");
				if (iKeyCount == 0)
					break;//nothing to send
				//write data to all connections
				ResetEnum();
				while ((iIndex = GetNextIndex()) != -1)
				{
					iLength = WriteSocket(rgiSocks[iIndex], rgcKeyBuf, iKeyCount, 0);
					if (iLength < 0)
					{
						Aprintf("Error on NetWrite from %d %d bytes: %s - closing connection\n",iIndex,iKeyCount, Err(iNetErrNo));
						ReleaseSocket(rgiSocks[iIndex]);
						rgiSocks[iIndex] = 0;
					}
				}
				iKeyCount = 0;
				break;
			}
		}
	}
	exit:
	//release all sockets
	ResetEnum();
	while ((iIndex = GetNextIndex()) != -1)
		ReleaseSocket(rgiSocks[iIndex]);
	ReleaseSocket(iListenSock);
}

/*
Open a TCP connection to the specified host and the default CHAT port.

Argument:
	pHostName - A pointer to the name of the host we want to connect to

Returns:
	A descriptor of the newly created socket.
*/
int ConnectTo(char* pcHostName)
{
	int iSock; //descriptor

	memset(&sNetAddr, 0, sizeof(NET_ADDR));

	sNetAddr.wRemotePort = CHAT_PORT;

	if ((sNetAddr.dwRemoteHost = ResolveName(pcHostName, 0, 0)) == 0)
		Aprintf("Error on ResolveName: %s\n",Err(iNetErrNo));
	else if ((iSock = GetSocket()) < 0)
		Aprintf("Error on GetSocket(): %s\n",Err(iNetErrNo));
	else if (SetSocketOption(iSock, 0, NET_OPT_NON_BLOCKING, 1, 1) < 0)
		Aprintf("Error on SetSocketOption(): %s\n",Err(iNetErrNo));
	else if (ConnectSocket(iSock, STREAM, &sNetAddr) < 0)
		Aprintf("Error on ConnectSocket: %s\n",Err(iNetErrNo));
	else
	{		//all well
		Aprintf("Trying to connect to %s (%s)\n",pcHostName, WriteName(&sNetAddr));
		return iSock;
	}
	Aprintf("Error on connect to %s - no con\n", pcHostName);
	return 0;
}

/*
Start a TCP server on spcified port.
(Returns immediately)

Argument:
	iPort - The TCP port number to listen on.

Returns:
	A descriptor of the server socket, or 0 if an error occured.
*/
int StartListen(int iPort)
{
	int iSock;

	memset(&sNetAddr, 0, sizeof(NET_ADDR));
	sNetAddr.wLocalPort = iPort;

	if ((iSock = GetSocket()) < 0)
		Aprintf("Error on serv GetSocket(): %s\n",Err(iNetErrNo));
	else if (SetSocketOption(iSock, 0, NET_OPT_NON_BLOCKING, 1, 1) < 0)
		Aprintf("Error on serv setOpt(): %s\n",Err(iNetErrNo));
	else if (ListenSocket(iSock, STREAM, &sNetAddr) < 0)
		Aprintf("Error on serv net_listen: %s\n",Err(iNetErrNo));
	else
	{
//		printf("Server sock successfully created\n");
		return iSock;
	}
	return 0;
}


/*
Creates a human understandable string from a Sockets error code.

Argument:
	uErrCode - The sockets error code.

Returns:
	A pointer to the (static) string representation of the error.
*/
char *Err(unsigned uErrCode)
{
	static char rgcUnk[30];
	static char *rgszErrs[] =
	{
		"NoErr",
		"InUse",
		"DOSErr",
		"NoMem",
		"NotNetconn",
		"IllegalOp",
		"BadPkt",
		"NoHost",
		"CantOpen",
		"NetUnreachable",
		"HostUnreachable",
		"ProtUnreachable",
		"PortUnreachable",
		"TimeOut",
		"HostUnknown",
		"NoServers",
		"ServerErr",
		"BadFormat",
		"BadArg",
		"EOF",
		"Reset",
		"WouldBlock",
		"UnBound",
		"NoDesc",
		"BadSysCall",
		"CantBroadcast",
		"NotEstab",
		"ReEntry",
	};

	if (uErrCode == ERR_API_NOT_LOADED)
		return "Sockets API not loaded";
	if ((uErrCode & 0xff) > ERR_RE_ENTRY)
	{
		sprintf(rgcUnk,"Unknown error 0x%04X",uErrCode);
		return rgcUnk;
	}
	return rgszErrs[uErrCode & 0xff];
}


/*
Show data on screen.

For now we just use vprintf to print the data and reprint any stuff that was
being edited.

Later we may put each person's data in a seperate block on screen, or whatever
seems fancy.

Arguments:
	Format and eclipse - Exactly the same as for prinf().
*/
void Aprintf (char *pFormat, ...)
{
	va_list pArgs;

	if (iKeyCount)//data been edited, start on newline
		printf("\n");

	va_start(pArgs, pFormat);
	vprintf(pFormat, pArgs);
	va_end(pArgs);

	//print the old edited stuff (if any)
	if (iKeyCount)
	{
		rgcKeyBuf[iKeyCount] = 0;
		printf("%s", rgcKeyBuf);
	}
}


/*
Create a string representation of the IP address and port, in the form
	a.b.c.d:port, eg 196.10.180.3:1400.
Max length is 22 bytes.

Arguments:
	psAddr - pointer to NET_ADDR structure containing address of host.

Returns:
	Pointer to (static) array containing null-terminated string.
*/
static char *WriteName(NET_ADDR *psAddr)
{
	static char rgcName[22];

	sprintf(rgcName, "%u.%u.%u.%u:%u",
		((BYTE *)&psAddr->dwRemoteHost)[0],
		((BYTE *)&psAddr->dwRemoteHost)[1],
		((BYTE *)&psAddr->dwRemoteHost)[2],
		((BYTE *)&psAddr->dwRemoteHost)[3],
		psAddr->wRemotePort
	);
	return rgcName;
}


/*
The next two functions implements an enumerator. To initialize (or reset),
	call resetEnum. Successive calls to getNextIndex will return all the used
	indexes. When there is no more used indexes, -1 is returned
*/
static int iPrev = -1;//privately used by next to functions
static void ResetEnum(void)
{
	iPrev = -1;
}

/*
Gets the next not 0 entry in the rgiSocks table.

Returns:
	The index of the next entry or -1 if no more.
*/
static int GetNextIndex(void)
{
	while (++iPrev < MAX_CONS)
	{
		if (rgiSocks[iPrev] != 0)
			return iPrev;
	}
	return -1;
}


/*
Gets the first not-assigned entry in the rgiSocks array.
Returns:
	The first not 0 index or -1 if all in use.
*/
static int GetFirstOpen(void)
{
	int iRet = 0;
	while (iRet < MAX_CONS)
	{
		if (rgiSocks[iRet] == 0)
			return iRet;
		iRet++;
	}
	return -1;
}

